/*-----------------------------------------------------------------
 *	$Id: x2sys_put.c,v 1.28 2008/04/03 01:58:21 guru Exp $
 *
 *      Copyright (c) 1999-2008 by P. Wessel
 *      See COPYING file for copying and redistribution conditions.
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; version 2 of the License.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      Contact info: gmt.soest.hawaii.edu
 *--------------------------------------------------------------------*/
/* x2sys_put will read one track bin file (.tbf) (from x2sys_binlist)
 * and update the track index database. The tbf file has the index
 * numbers of all the x by x degree boxes traversed by each data set.
 * The bin dimension and extent are determined by the tbf header.
 * This bin info is added to the track index database. This database
 * is then used by program x2sys_get that locates all the data track
 * within a given area that optionally contain specific data columns.
 *
 * Author:	Paul Wessel
 * Date:	14-JUN-2004
 * Version:	1.1, based on the spirit of the old mgg code
 *
 */

#include "x2sys.h"

int main (int argc, char **argv)
{
	char *TAG = CNULL;

	struct X2SYS_INFO *s;
	struct X2SYS_BIX B;

	struct X2SYS_BIX_TRACK_INFO *this_info, *new_info;
	struct X2SYS_BIX_TRACK *this_track;

	char track[GMT_TEXT_LEN], line[BUFSIZ];
	char track_file[BUFSIZ], index_file[BUFSIZ], old_track_file[BUFSIZ], old_index_file[BUFSIZ];
	char track_path[BUFSIZ], index_path[BUFSIZ], old_track_path[BUFSIZ], old_index_path[BUFSIZ];

	BOOLEAN error = FALSE, replace = FALSE, delete = FALSE, found_it, swap = FALSE;

	FILE *fp = NULL, *fbin, *ftrack;

	int index, id, bin, free_id, max_flag, flag;
	int i, last_id, bit, total_flag;

	int x2sys_bix_remove_track (int track_id, struct X2SYS_BIX *B);

	argc = GMT_begin (argc, argv);
	
	for (i = strlen(argv[0]); i >= 0 && argv[0][i] != '/'; i--);
	X2SYS_program = &argv[0][i+1];	/* Name without full path */

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				case 'D':	/* Remove all traces of these tracks from the database */
					delete = TRUE;
					replace = TRUE;
					break;
				case 'V':
				case '\0':
					error += GMT_parse_common_options (argv[i], NULL, NULL, NULL, NULL);
					break;
				case 'T':
					TAG = &argv[i][2];
					break;
				case 'S':
					swap = TRUE;	/* Swap option for index.b reading */
					break;
				default:
					error = TRUE;
					fprintf (stderr, "%s: Unrecognized option -%c\n", GMT_program, argv[i][1]);
					break;
			}
		}
		else if ((fp = GMT_fopen (argv[i], "r")) == NULL) {
			fprintf (stderr, "%s: ERROR: Could not open file %s\n", GMT_program, argv[i]);
			exit (EXIT_FAILURE);
		}
	}

	if (argc == 1 || error || GMT_give_synopsis_and_exit) {
		fprintf (stderr, "x2sys_put %s - Update track index database from track bin file\n\n", X2SYS_VERSION);
		fprintf (stderr, "usage: x2sys_put [<info.tbf>]  -T<TAG> [-S] [-V]\n\n");
		fprintf (stderr, "	<info.tbf> is one track bin file from x2sys_binlist [Default reads stdin]\n");
		fprintf (stderr, "	-T <TAG> is the system tag for this compilation\n");

		if (GMT_give_synopsis_and_exit) exit (EXIT_FAILURE);

		fprintf (stderr, "\n\tOPTIONS:\n");
		fprintf(stderr,"	-D will remove the listed tracks  [Default will add to database]\n");
		fprintf (stderr, "	-S Byte swap binary records during read [no swapping]\n");
		GMT_explain_option ('V');
		exit (EXIT_FAILURE);
	}

	if (fp == NULL) fp = GMT_stdin;	/* Read stdin instead */

	GMT_fgets (line, BUFSIZ, fp);	/* Got the first record from the track binindex file */
	if (strncmp (&line[2], TAG, strlen(TAG))) {
		fprintf (stderr,"%s: The TAG specified (%s) does not match the one in the .tbf file (%s)\n", GMT_program, TAG, &line[2]);
		exit (EXIT_FAILURE);
	}

	/* Open TAG file and set the operational parameters */

	x2sys_err_fail (x2sys_set_system (TAG, &s, &B, &GMT_io), TAG);

	for (i = max_flag = 0, bit = 1; i < s->n_fields; i++, bit <<= 1) max_flag |= bit;

	x2sys_bix_init (&B, FALSE);

	sprintf (track_file, "%s_tracks.d", TAG);
	sprintf (index_file, "%s_index.b",  TAG);
	x2sys_path (track_file, track_path);
	x2sys_path (index_file, index_path);

	if ((ftrack = GMT_fopen (track_path, "r")) == NULL) {
		fprintf (stderr, "%s: Could not find %s\n", GMT_program, track_file);
		exit (EXIT_FAILURE);
	}

	/* Read existing track-information from <ID>_tracks.d file */

	 x2sys_err_fail (x2sys_bix_read_tracks (TAG, &B, 0, &last_id), "");
	 last_id--;

	/* Read geographical track-info from <ID>_index.b file */

	x2sys_err_fail (x2sys_bix_read_index (TAG, &B, swap), "");

	/* Ok, now we can start reading new info */

#ifdef DEBUG
	GMT_memtrack_off (GMT_mem_keeper);
#endif
	GMT_fgets (line, BUFSIZ, fp);
	while (line[0] == '>') {	/* Next segment */
		sscanf (line, "> %s", track);
		for (this_info = B.head; this_info->next_info && strcmp (this_info->next_info->trackname, track) < 0; this_info = this_info->next_info);
		free_id = 0;
		found_it = (this_info->next_info != (struct X2SYS_BIX_TRACK_INFO *)NULL);
		if (found_it && !strcmp (this_info->next_info->trackname, track)) {
			if (gmtdefs.verbose) fprintf (stderr, "%s: Removing existing information for track: %s\n", GMT_program, track);
			free_id = x2sys_bix_remove_track (this_info->next_info->track_id, &B);
		}
		if (delete) {	/* Give message and go back and read next track */
			if (found_it) {
				fprintf (stderr, "%s: track %s removed\n", GMT_program, track);
				this_info->next_info = this_info->next_info->next_info;
			}
			else
				fprintf (stderr, "%s: track %s was not found in the database!\n", GMT_program, track);
		}

		/* Read the tbf information for this track */

		if (delete) {	/* Just wind past this segment */
			while (line[0] != '>') GMT_fgets (line, BUFSIZ, fp);
		}
		else {
			if (gmtdefs.verbose) fprintf (stderr, "%s: Adding track: %s\n", GMT_program, track);

			/* If a track is replaced, then use the same id_no, else increment to get a new one */

			id = (free_id) ? free_id : ++last_id;
			if (!free_id) {
				new_info = x2sys_bix_make_entry (track, id, 0);
				new_info->next_info = this_info->next_info;
				this_info->next_info = new_info;
				this_info = new_info;
			}
			else
				this_info = this_info->next_info;

			total_flag = 0;
			while (GMT_fgets (line, BUFSIZ, fp) && line[0] != '>') {
				sscanf (line, "%*s %*s %d %d", &index, &flag);
				if (flag > max_flag) {
					fprintf (stderr, "%s: data flag (%d) exceed maximum (%d) for track %s!\n", GMT_program, flag, max_flag, track);
					exit (EXIT_FAILURE);
				}
				if (B.base[index].n_tracks == 0) {
					B.base[index].first_track = x2sys_bix_make_track (0, 0);
					B.base[index].last_track = B.base[index].first_track;
				}
				B.base[index].last_track->next_track = x2sys_bix_make_track (id, flag);
				B.base[index].last_track = B.base[index].last_track->next_track;
				B.base[index].n_tracks++;
				total_flag |= flag;
			}
			this_info->flag = total_flag;
		}
	}
	if (fp != GMT_stdin) GMT_fclose (fp);
#ifdef DEBUG
	GMT_memtrack_on (GMT_mem_keeper);
#endif

	/* Done, now we must rewrite the <ID>_index.b and <ID>_tracks.d files */

	sprintf (old_track_file, "%s_old", track_file);
	sprintf (old_index_file, "%s_old", index_file);
	x2sys_path (old_track_file, old_track_path);
	x2sys_path (old_index_file, old_index_path);

	if (rename (track_path, old_track_path)) {
		fprintf (stderr, "%s: Rename failed for %s. Aborting!\n", GMT_program, track_path);
		exit (EXIT_FAILURE);
	}
	if (rename (index_path, old_index_path)) {
		fprintf (stderr, "%s: Rename failed for %s. Aborts!\n", GMT_program, index_path);
		exit (EXIT_FAILURE);
	}

	if ((ftrack = fopen (track_path, "w")) == NULL) {
		fprintf (stderr, "%s: Failed to create %s. Aborts!\n", GMT_program, track_path);
		exit (EXIT_FAILURE);
	}
	if ((fbin = fopen (index_path, "wb")) == NULL) {
		fprintf (stderr, "%s: Failed to create %s. Aborts!\n", GMT_program, index_path);
		exit (EXIT_FAILURE);
	}
	fprintf (ftrack,"# %s\n", TAG);
	for (this_info = B.head->next_info; this_info; this_info = this_info->next_info)
		fprintf (ftrack,"%s %d %d\n",this_info->trackname, this_info->track_id, this_info->flag);

	GMT_fclose (ftrack);
	chmod (track_file, (mode_t)S_RDONLY);

	for (bin = 0; bin < B.nm_bin; bin++) {
		if (B.base[bin].n_tracks == 0) continue;

		fwrite ((void *)(&bin), (size_t)4, (size_t)1, fbin);
		fwrite ((void *)(&B.base[bin].n_tracks), (size_t)4, (size_t)1, fbin);
		for (this_track = B.base[bin].first_track->next_track; this_track; this_track = this_track->next_track) {
			fwrite ((void *)(&this_track->track_id), (size_t)4, (size_t)1, fbin);
			fwrite ((void *)(&this_track->track_flag), (size_t)4, (size_t)1, fbin);
		}
	}
	GMT_fclose (fbin);
	chmod (index_file, (mode_t)S_RDONLY);

	if (gmtdefs.verbose) fprintf (stderr, "%s completed successfully\n", GMT_program);

	x2sys_end (s);

	GMT_end (argc, argv);

	exit (EXIT_SUCCESS);
}

int x2sys_bix_remove_track (int track_id, struct X2SYS_BIX *B)
{
	/* Remove all traces of the track with id track_id from structure tree */

	struct X2SYS_BIX_TRACK *track, *skip_track;
	int bin;

	for (bin = 0; bin < B->nm_bin; bin++) {
		if (B->base[bin].n_tracks == 0) continue;	/* No tracks crossed this bin */

		for (track = B->base[bin].first_track; track->next_track && track->next_track->track_id != track_id; track = track->next_track);	/* Finds the track or end-of-list */

		if (track->next_track) {	/* Ok, found it. Lets remove it from the list */
			skip_track = track->next_track;
			track->next_track = skip_track->next_track;
			skip_track->next_track = NULL;
			B->base[bin].n_tracks--;
			if (!track->next_track) B->base[bin].last_track = track;
			GMT_free ((void *) skip_track);
			if (B->base[bin].n_tracks == 0) GMT_free ((void *)B->base[bin].first_track);
		}
	}
	return (track_id);
}
