/*--------------------------------------------------------------------
 *    $Id: ps2raster.c,v 1.11 2006/09/08 17:35:52 remko Exp $
 *
 *    Copyright (c) 2006 by J. Luis
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 *
 * ps2raster converts one or several ps file(s) to other formats using GhostScript.
 * It works by modifying the page size in order that the image will have a size
 * which is specified by the BoundingBox. 
 * As an option, a tight BoundingBox may be computed.
 * ps2raster uses the ideas of the EPS2XXX.m from Primoz Cermelj published in MatLab Central
 * and of psbox_sh of Remko Scharroo.
 *
 *
 *--------------------------------------------------------------------*/
/*
 * Author:	Joaquim Luis
 * Created:	15-FEB-2005
 * Modified:	15-NOV-2005 (It used to fail too many times in detecting if is GMT_ps)
 *		01-DEC-2005 Changed the generated script name to not overlap with -N
 *		27-DEC-2005 Integrated into GMT's misc supplement
 *		07-SEP-2006 Added EPS format; no more GMT specific; high-quality PDF output
 */

#include "gmt.h"

int main (int argc, char **argv) {
	int i, j, k, len, x0 = 0, x1 = 612, y0 = 0, y1 = 828, dpi = 0, w = 828, h = 612, n_files = 0, n_alloc = 100;
	int error = FALSE, adjust_BB = FALSE, no_rem = FALSE, have_list = FALSE, verbose = FALSE, do_eps = FALSE;
	int got_BB, got_adjusted_BB, atend, setup;
	
	char ps_file[BUFSIZ];
	char **ps_names;
	char line[BUFSIZ], c[BUFSIZ], c1[20], c2[20], c3[20], c4[20], c5[20], cmd[BUFSIZ];
	char *tmp_file = NULL, *out_file = NULL, *BB_file = NULL, *fullGsPath = NULL;
	char *script_name = NULL, *scriptBB = NULL, *gs_params = NULL, *gs_BB = NULL;
	char *ext = ".jpg", *device = "jpeg";
	FILE *fp = NULL, *fpo = NULL, *fpb = NULL, *fpl = NULL;
	void run_cmd (char *name, char *cmd, int rem1, int rem2);
	
	/* Parameters for all the formats available */
	gs_params = "-q -dNOPAUSE -dBATCH -dUseFlateCompression=true -dPDFSETTINGS=/prepress -dEmbedAllFonts=true -dMonoImageFilter=/FlateEncode -dAutoFilterGrayImages=false -dGrayImageFilter=/FlateEncode -dAutoFilterColorImages=false -dColorImageFilter=/FlateEncode";
	gs_BB = "-q -dNOPAUSE -dBATCH -sDEVICE=bbox";

#ifdef _WIN32
	fullGsPath = "gswin32c";
	script_name = "psTOraster.bat";
	scriptBB = "to_BB.bat";
#else
	fullGsPath = "gs";
	script_name = "psTOraster.sc";
	scriptBB = "to_BB.sc";
#endif
	
	/* Check and interpret the command line arguments */
	
	GMT_program = argv[0];
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch(argv[i][1]) {
				case '\0':
					GMT_give_synopsis_and_exit = TRUE;
					break;
				
				/* Supplemental parameters */
				case 'A':
					adjust_BB = TRUE;
					break;
				case 'G':
					fullGsPath = &argv[i][2];
					break;
				case 'E':
                                        dpi = atoi(&argv[i][2]);
					break;
				case 'L':
					if ((fpl = fopen (&argv[i][2], "r")) == NULL) {
						fprintf (stderr, "%s: GMT ERROR: Cannot to open list file %s\n", GMT_program, argv[i]); 
						error = TRUE;
					}
					have_list = TRUE;
					break;
				case 'N':
					no_rem = TRUE;
					break;
				case 'T':	/* Select output format */
					for (j = 2; argv[i][j]; j++) {
						switch (argv[i][j]) {
							case 'e':	/* EPS */
								ext = ".eps";
								do_eps = TRUE;
								break;
							case 'f':	/* PDF */
								ext = ".pdf";
								device = "pdfwrite";
								break;
							case 'j':	/* JPEG */
								ext = ".jpg";
								device = "jpeg";
								break;
							case 'g':	/* PNG */
								ext = ".png";
								device = "png16m";
								break;
							case 'm':	/* PPM */
								ext = ".ppm";
								device = "ppm16m";
								break;
							case 't':	/* TIFF */
								ext = ".tif";
								device = "tifflzw";
								break;
							default:
								fprintf (stderr, "%s: GMT ERROR: Unrecognized option %s\n", GMT_program, argv[i]); 
								error = TRUE;
								break;
						}
					}
					break;
				case 'V':
					verbose = TRUE;
					break;
				/* Options not recognized */
				default:
					error = TRUE;
					break;
			}
		}
		else {
			strcpy (ps_file, argv[i]);
			n_files++;
		}
	}
	
	if (argc == 1 || GMT_give_synopsis_and_exit || error) {
		fprintf (stderr,"ps2raster %s - Converts one or several PS file(s) to raster formats using GhostScript.\n\n", GMT_VERSION);
		fprintf(stderr,"usage: ps2raster <psfile1> <psfile2> <...> [-A] [-E<resolution>]\n");
		fprintf(stderr,"       [-G<ghost_path>] [-L<listfile>] [-N] [-Tf|j|g|m|t] [-V]\n\n");
		if (GMT_give_synopsis_and_exit) exit (EXIT_FAILURE);
		
		fprintf (stderr,"Works by modifying the page size in order that the resulting\n");
		fprintf (stderr,"image will have the size specified by the BoundingBox.\n");
		fprintf (stderr,"As an option, a tight BoundingBox may be computed.\n\n");
		fprintf (stderr,"	<psfile(s)> postscript file(s) to be converted.\n");
		fprintf(stderr,"\n\tOPTIONS:\n");
		fprintf (stderr,"\t-A Adjust the BoundingBox to the minimum required by the image contents\n");
		fprintf (stderr,"\t-E Set raster resolution in dpi [default = 720 for PDF, 300 for others]\n");
		fprintf (stderr,"\t-G Full path to your ghostscript executable.\n");
		fprintf (stderr,"\t   NOTE: Under Unix systems this is generally not necessary.\n");
		fprintf (stderr,"\t   Under Windows, ghostscript is not added to the system's path.\n");
		fprintf (stderr,"\t   So either you do it yourself, or give the full path here.\n");
		fprintf (stderr,"\t   (e.g. -Gc:\\programs\\gs\\gs7.05\\bin\\gswin32c).\n");
		fprintf (stderr,"\t-L The <listfile> is an ASCII file with names of ps files to be converted\n");
		fprintf (stderr,"\t-N Do Not remove auxiliary files [default, does]. Auxiliary files are\n");
		fprintf (stderr,"\t   build using as base the input ps files and consist of:\n");
		fprintf (stderr,"\t   psfile_tmpxx.eps -> postcript with a modified BoundingBox\n");
		fprintf (stderr,"\t   ps2raster.sc|bat -> script (or batch) with the GS command that does the job\n");
		fprintf (stderr,"\t   Use this option to save the script and run it later with different settings\n");
		fprintf (stderr,"\t-T Set output raster format [default is jpeg]\n");
		fprintf (stderr,"\t   e means EPS\n");
		fprintf (stderr,"\t   f means PDF\n");
		fprintf (stderr,"\t   j means JPEG\n");
		fprintf (stderr,"\t   g means PNG\n");
		fprintf (stderr,"\t   m means PPM\n");
		fprintf (stderr,"\t   t means TIF\n");
		exit (EXIT_FAILURE);
	}

	if (n_files > 1 && have_list) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR: Cannot handle both a file list and multiple ps files in input\n", GMT_program);
		error = TRUE;
	}

	if (error) exit (EXIT_FAILURE);

	/* Use default DPI if not already set */
	if (dpi <= 0) {
		dpi = 300;
		if (!strcmp(ext,".pdf")) dpi = 720;
	}

	/* Multiple files in a file with their names */
	if (have_list) {
		ps_names = (char **) calloc ((size_t)(n_alloc), sizeof (char *));
		while (fgets (line, BUFSIZ, fpl) != NULL) {
			ps_names[n_files] = (char *) calloc (BUFSIZ, sizeof (char));
			if (line[0] == '#' || line[0] == '\n' || line[0] == '\r') continue;
			sscanf (line, "%s",ps_names[n_files]);
			n_files++;
			if (n_files > n_alloc) {
				n_alloc += 100;
				ps_names = realloc (ps_names, n_alloc * sizeof (char *));
			}
		}
		fclose (fpl);
	}

	/* Multiple files given on command line */
	else if (n_files > 1) {	
		ps_names = (char **) calloc ((size_t)(n_alloc), sizeof (char *));
		j = 0;
		for (k = 1; k < argc; k++) {
			if (argv[k][0] == '-') continue;
			ps_names[j] = (char *) calloc (BUFSIZ, sizeof (char));
			ps_names[j] = argv[k];
			j++;
			if (n_files > n_alloc) {
				n_alloc += 100;
				ps_names = realloc (ps_names, n_alloc * sizeof (char *));
			}
		}
	}
	else {				/* Single file */
		ps_names = (char **) calloc ((size_t)(1), sizeof (char *));
		ps_names[0] = ps_file;
	}

	/* Loop over all input files */
	for (k = 0; k < n_files; k++) {
		strcpy(ps_file,ps_names[k]);
		if ((fp = fopen (ps_file, "r")) == NULL) {
			fprintf (stderr, "%s: Cannot to open file %s\n", GMT_program, ps_file);
			continue;
		}	
		atend = FALSE;
		got_BB = FALSE;
		got_adjusted_BB = FALSE;
		setup = FALSE;

		len = strlen(ps_file);
		j = len - 1;
		for (i = 0; i < len; i++, j--) {
			if (ps_file[j] == '.') break;
		} 
		tmp_file = (char *) calloc ((size_t)(j+12), sizeof(char));
		out_file = (char *) calloc ((size_t)(j+6), sizeof(char));

		/* Adjust to a tight BoundingBox if User required so */
		if (adjust_BB) {
			BB_file = (char *) calloc ((size_t)(j+7), sizeof(char));
			for (i = 0; i < j; i++) BB_file[i] = ps_file[i];
			strcat(BB_file,".bbox");
			sprintf(cmd,"%s %s %s 2> %s",fullGsPath,gs_BB,ps_file,BB_file);
			run_cmd(scriptBB, cmd, 0, 0);	/* Execute the script that computes the tight BB */
			if ((fpb = fopen (BB_file, "r")) == NULL) {
				fprintf (stderr, "%s: Unable to open file %s\n", GMT_program, BB_file);
				exit (EXIT_FAILURE);
			}
			while (fgets (line, BUFSIZ, fpb) != NULL) {
				sscanf (line, "%s",c); 
				if (strcmp(c,"%%BoundingBox:") == 0) {
					sscanf (line, "%s %s %s %s %s",c1,c2,c3,c4,c5); 
					x0 = atoi (c2);		y0 = atoi (c3);
					x1 = atoi (c4);		y1 = atoi (c5);
					got_adjusted_BB = TRUE;
				}
			}
			fclose (fpb);
			run_cmd(BB_file, cmd, 1, 0);	/* Remove the file with BB info */
			free ((void *)BB_file);
		}

		/* Open temporary file to be processed by ghostscript. When just creating EPS, tmp_file is for keeps */
		for (i = 0; i < j; i++) tmp_file[i] = ps_file[i];
		for (i = 0; i < j; i++) out_file[i] = ps_file[i];
		if (do_eps)
			strcat(tmp_file,ext);
		else
			strcat(tmp_file,"_tmpxx.eps");
		if ((fpo = fopen (tmp_file, "w")) == NULL) {
			fprintf (stderr, "%s: Unable to open file %s for writing\n", GMT_program, tmp_file);
			exit (EXIT_FAILURE);
		}
		strcat(out_file,ext);

		/* Scan input file for BoundingBox statements */
		i = 0;
		while (fgets (line, BUFSIZ, fp) != NULL) {
			i++;
			if (line[0] != '%') {	/* Copy any non-comment line, except one containing /PageSize in the Setup block */
				if (setup && strstr(line,"/PageSize") != NULL) continue;
				fprintf (fpo, "%s", line);
				continue;
			}
			sscanf (line, "%s",c); 
			if ((!got_BB || atend) && !strncmp(c, "%%BoundingBox:", 14)) {
				if (atend) continue;	/* Simply skip second BoundingBox */
				sscanf (line, "%s %s %s %s %s",c1,c2,c3,c4,c5); 
				if (!strncmp(c2,"(atend)", 7)) {
					if (!adjust_BB) {
						fprintf (stderr, "%s: To convert the %s file you must use the -A option\n", GMT_program, ps_file);
						exit (EXIT_FAILURE); 
					}
					atend = TRUE;
				}
				else if (!got_adjusted_BB) {
					x0 = atoi (c2);		y0 = atoi (c3);
					x1 = atoi (c4);		y1 = atoi (c5);
				}
				w = irint (fabs(x1 - x0));
				h = irint (fabs(y1 - y0));
				if (do_eps)
					fprintf (fpo, "%s %d %d %d %d\n", c1,x0,y0,x1,y1);
				else {
					fprintf (fpo, "%s 0 0 %d %d\n", c1,w,h);
					fprintf (fpo, "%d %d translate\n", -x0,-y0);
				}
				got_BB = TRUE;
				continue;
			}
			if (!strncmp(c, "%%BeginSetup", 12)) setup=TRUE;
			if (!strncmp(c, "%%EndSetup", 10)) setup=FALSE;
			fprintf (fpo, "%s", line);

			if (!got_BB && i >= 20) {
				fprintf (stderr, "%s: GMT FATAL ERROR: The file %s has no BoundingBox in the first 20 lines\n", GMT_program, ps_file);
				exit (EXIT_FAILURE); 
			}
		}

		if (!got_BB) {
			fprintf (stderr, "%s: GMT FATAL ERROR: The file %s has no BoundingBox\n", GMT_program, ps_file);
			exit (EXIT_FAILURE); 
		}

		fclose (fpo);
		fclose (fp);

		/* Build the converting ghost command and execute it */
		if (!do_eps) {
			w = (int)rint (w / 72.0 * dpi);
			h = (int)rint (h / 72.0 * dpi);
			sprintf(cmd,"%s %s -sDEVICE=%s -g%dx%d -r%d -sOutputFile=%s -f%s",fullGsPath,gs_params,device,w,h,dpi,out_file,tmp_file);
			run_cmd(script_name, cmd, 0, no_rem);
			if (!no_rem) run_cmd(tmp_file, cmd, 1, 0);
		}

		free ((void *)tmp_file);
		free ((void *)out_file);
	}

	free ((void *)ps_names);

	exit (EXIT_SUCCESS);
}

void run_cmd(char *name, char *cmd, int rem1, int rem2) {
	FILE *fp;

	if (rem1) {
#ifdef _WIN32
		sprintf (cmd, "del %s\n", name);
#else
		sprintf (cmd, "rm -f %s\n", name);
#endif
		system(cmd);
		return;
	}

	fp = fopen (name, "w");
#ifdef _WIN32
	fprintf (fp, "echo off\n");
#endif
	fprintf (fp, "%s\n", cmd);
	fclose (fp);

#ifdef _WIN32
	sprintf (cmd, "%s",name);
	system(cmd);
#else
	sprintf (cmd, "sh %s", name);
	system(cmd);
#endif
	if (!rem2) {
#ifdef _WIN32
		sprintf (cmd, "del %s\n", name);
#else
		sprintf (cmd, "rm -f %s\n", name);
#endif
		system(cmd);
	}
}
